/* Copyright (C) 2015-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef OPTIONALAUTOPTR_H_42349f72_b718_4ba2_bb10_e1a93695437d
#define OPTIONALAUTOPTR_H_42349f72_b718_4ba2_bb10_e1a93695437d

#include <vnccommon/Optional.h>
#include <vnccommon/OptionalReference.h>

#include <memory>

/**
 * \cond VNCCOMMON
 */
namespace vnccommon
{

/**
 * \brief A container holding an \c auto_ptr for an optional value.
 */
template <typename T>
class OptionalAutoPtr
{
public:
    /**
     * \brief Default constructor.
     *
     * Creates an \ref OptionalAutoPtr representing 'nothing'.
     */
    OptionalAutoPtr()
    {
    }

    /**
     * \brief 'Nothing' constructor.
     *
     * Creates an \ref OptionalAutoPtr representing 'nothing'.
     */
    OptionalAutoPtr(NoneType)
    {
    }

    /**
     * \brief 'Something' constructor
     *
     * Creates an \ref OptionalAutoPtr representing 'something'.
     *
     * \note This takes ownership of the \c auto_ptr.
     *
     * \param value The value the \ref OptionalAutoPtr object will hold.
     */
    explicit OptionalAutoPtr(std::auto_ptr<T> value)
        : mValue(value)
    {
    }

    /**
     * \brief Copy constructor
     *
     * \note This takes ownership of the \c auto_ptr in \a o.
     *
     * \param o The \ref OptionalAutoPtr to take the value from.
     */
    OptionalAutoPtr(const OptionalAutoPtr<T>& o)
        : mValue(o.mValue)
    {
    }

    /**
     * \brief Assignment operator
     *
     * \note This takes ownership of the \c auto_ptr in \a o
     *
     * \param o The \ref OptionalAutoPtr to take the value from.
     *
     * \return This object.
     */
    OptionalAutoPtr<T>& operator=(const OptionalAutoPtr<T>& o)
    {
        mValue = o.mValue;
        return *this;
    }

    /**
     * \brief Deletes the current value and replaces it with the one provided.
     *
     * \note This takes ownership of the \c auto_ptr.
     *
     * \param value The value the \ref OptionalAutoPtr object will hold.
     */
    void reset(std::auto_ptr<T> value)
    {
        mValue.reset(value.release());
    }

    /**
     * \brief Deletes the current value.
     */
    void reset()
    {
        mValue.reset();
    }

    /**
     * \brief Queries whether the \ref OptionalAutoPtr has a value.
     *
     * \return Whether the \ref OptionalAutoPtr has a value.
     */
    bool hasValue() const
    {
        return mValue.get() != NULL;
    }

    /**
     * \brief Gets the value.
     *
     * \return The value.
     */
    const T& value() const
    {
        if (!hasValue())
        {
            throw std::logic_error(
                    "Attempted to retrieve value of empty OptionalAutoPtr");
        }

        return *mValue;
    }

    /**
     * \brief Gets the value.
     *
     * \return The value.
     */
    T& value()
    {
        if (!hasValue())
        {
            throw std::logic_error(
                    "Attempted to retrieve value of empty OptionalAutoPtr");
        }

        return *mValue;
    }

    /**
     * \brief Releases ownership of the \c auto_ptr.
     *
     * \return The \c auto_ptr.
     */
    std::auto_ptr<T> release()
    {
        return mValue;
    }

    /**
     * \brief Returns an \ref OptionalReference to the value.
     *
     * \return A reference to the value.
     */
    OptionalReference<T> getReference()
    {
        if(hasValue())
        {
            return OptionalReference<T>(*mValue.get());
        }
        else
        {
            return OptionalReference<T>();
        }
    }

    /**
     * \brief Returns an \ref OptionalReference to the value.
     *
     * \return A reference to the value.
     */
    OptionalReference<const T> getReference() const
    {
        if(hasValue())
        {
            return OptionalReference<const T>(*mValue.get());
        }
        else
        {
            return OptionalReference<const T>();
        }
    }

    /**
     * \brief Throws the specified exception if no value is present.
     */
    template<typename E>
    const T& orThrow(const E& e) const
    {
        if(!hasValue())
        {
            throw e;
        }

        return value();
    }

    /**
     * \brief Throws the specified exception if no value is present.
     */
    template<typename E>
    T& orThrow(const E& e)
    {
        if(!hasValue())
        {
            throw e;
        }

        return value();
    }

    /**
     * \brief Throws the specified exception if no value is present.
     */
    template<typename E>
    std::auto_ptr<T> releaseOrThrow(const E& e)
    {
        if(!hasValue())
        {
            throw e;
        }

        return release();
    }

private:
    mutable std::auto_ptr<T> mValue;
};

/**
 * \brief Convenience function for constructing an \ref OptionalAutoPtr.
 *
 * \param value The value the \ref OptionalAutoPtr object will hold.
 * \return An \ref OptionalAutoPtr representing 'something'.
 */
template <typename T>
OptionalAutoPtr<T> MakeOptionalAutoPtr(std::auto_ptr<T> value)
{
    return OptionalAutoPtr<T>(value);
}

} // end of namespace vnccommon
/**
 * \endcond
 */

#endif // OPTIONALAUTOPTR_H_42349f72_b718_4ba2_bb10_e1a93695437d

